// *********************************************************************
//
// Example of subclassing a Metatrader chart window, in order to handle
// both keyboard events and mouse clicks. DO NOT UNDER ANY CIRCUMSTANCES
// USE THIS ON A LIVE ACCOUNT. Any code such as this has the potential
// to crash Metatrader. Not seriously tested, and not tested at all on
// anything other than 32-bit Windows XP. May require a recent-ish
// build of Metatrader. Only tested on build 220.
//
// This is a script file (not an EA) which needs to be put in 
// experts\scripts and then compiled.
//
// MQLSubclassExample.dll needs to be present in the experts\libraries
// folder (or in System32), and "Allow DLL imports" needs to be turned
// on.
//
// The code should display a series of text objects on the chart.
// You can then do the following:
//
//   * Buy at market, by pressing Ctrl+Shift+B or by holding down Alt
//     while double-clicking on the "Buy" text
//   * Sell at market, by pressing Ctrl+Shift+S or by holding down Alt
//     while double-clicking on the "Sell" text
//   * Close all pending and open orders for the symbol, by pressing
//     Ctrl+Shift+C.
//   * Increase the number of lots used for buys and sells, by
//     pressing Ctrl+Shift+Plus or by holding down Alt while double-clicking
//     on the up arrow.
//   * Decrease the number of lots used for buys and sells, by
//     pressing Ctrl+Shift+Minus or by holding down Alt while double-clicking
//     on the down arrow.
//   * Terminate the script by pressing Ctrl+Shift+X.
//
// The chart obviously has to be the active window before it will receive
// keyboard shortcuts (just as for native Metatrader shortcuts such as
// PageUp or PageDown).
//
// The initial lot size to use, and the step for changes, are defined by
// parameters for the script. You can also choose whether trading operations
// require confirmation or not. 
//
// The same general principles apply to using code such as this in an
// EA rather than a script. However, rather than doing a continuous
// loop with a Sleep() in it, you'd probably stick to the natural 
// mechanism of having start() called on each market tick. The slight
// issue with this is that keyboard shortcuts/mouse clicks wouldn't 
// be processed until the next market tick - potentially a couple of 
// days away if you try using them at weekends. However, it's possible
// to ignore very old events by looking at the event-time passed back
// in EventLocalTime[0] versus the current clock according to LocalTime().
//
// *********************************************************************

#property show_inputs



// *********************************************************************
// Import of the DLL which does the subclassing of the chart window,
// allowing the script to see keyboard and mouse events. The script
// will terminate without doing anything if the DLL is not present,
// either in experts\libraries or in System32.
// *********************************************************************

#import "MQLSubclassExample.dll"
   // Sets up subclassing on a chart window, using the hWnd obtained from WindowHandle(Symbol(), 0)
   bool SubclassWindow(int hWnd);
   
   // Removes subclassing, on termination - VERY IMPORTANT
   bool UnsubclassWindow(int hWnd);
   
   // Reads from the queue of keyboard/mouse events which have happened since the script 
   // last queried the queue
   bool GetFromQueue(int hWnd, int & EventType[], int & Param1[], int & Param2[], int & Param3[], int & EventLocalTime[]);
#import



// *********************************************************************
// External parameters 
// *********************************************************************

// Initial lot size used for orders 
extern double  InitialLotSize = 0.01;

// Step by which the lot size is changed when double-clicking on the arrows,
// or when using Ctrl+Shift+plus/minus
extern double  LotSizeStep = 0.01;

// Whether trading actions have to be confirmed
extern bool    ConfirmActions = true;



// *********************************************************************
// Definitions of the event types passed back by the subclassing in the DLL
// *********************************************************************

#define WM_KEYDOWN         256
#define WM_LBUTTONDBLCLK   515
#define WM_RBUTTONDBLCLK   518


// *********************************************************************
// Global variables 
// *********************************************************************

// Global variable which holds the current number of lots configured,
// starting at InitialLotSize and alterable by double-clicking on the
// up and down arrows or by using Ctrl+Shift and plus/minus.
double glbLotSizeInUse;



// *********************************************************************
// Initialisation and termination
// *********************************************************************

// Script initialisation
int init()
{
   if (!IsDllsAllowed()) {
      // Display error message later, in start()
   } else {
      // Store the initial lot size in a global variable, for later alteration
      glbLotSizeInUse = InitialLotSize;

      // Set up the subclassing on the window 
      if (!SubclassWindow(WindowHandle(Symbol(), 0))) {
         MessageBox("Failed to set up the subclassing of the chart window. Is MQLSubclassExample.dll correctly installed?");
      }
   
      // Create the OBJ_LABEL objects making up the pseudo-buttons which you can Alt+dblclick
      CreateObjects();
   }
   return(0);
}

// Script termination 
int deinit()
{
   // Remove the subclassing from the window - VERY IMPORTANT
   UnsubclassWindow(WindowHandle(Symbol(), 0));
   
   // Delete the OBJ_LABEL objects 
   RemoveObjects();
   return(0);
}


// *********************************************************************
// Display of chart objects (OBJ_LABEL)
// *********************************************************************

// Create the pseudo-button objects (on initialisation). If changing
// the font sizes, see the notes on the unavoidable cheat in the
// service function IsMouseWithinObjectCoordinates()
void CreateObjects()
{
   ObjectCreate("Buy button", OBJ_LABEL, 0, 0, 0);
   ObjectSet("Buy button", OBJPROP_XDISTANCE, 50);
   ObjectSet("Buy button", OBJPROP_YDISTANCE, 50);
   ObjectSetText("Buy button", "BUY", 20, "Arial", LightGreen);

   ObjectCreate("Sell button", OBJ_LABEL, 0, 0, 0);
   ObjectSet("Sell button", OBJPROP_XDISTANCE, 150);
   ObjectSet("Sell button", OBJPROP_YDISTANCE, 50);
   ObjectSetText("Sell button", "SELL", 20, "Arial", Red);

   ObjectCreate("Lot size", OBJ_LABEL, 0, 0, 0);
   ObjectSet("Lot size", OBJPROP_XDISTANCE, 50);
   ObjectSet("Lot size", OBJPROP_YDISTANCE, 100);
   DisplayLotSize();

   ObjectCreate("Lot size increase", OBJ_LABEL, 0, 0, 0);
   ObjectSet("Lot size increase", OBJPROP_XDISTANCE, 170);
   ObjectSet("Lot size increase", OBJPROP_YDISTANCE, 100);
   ObjectSetText("Lot size increase", "", 14, "Wingdings", Yellow);

   ObjectCreate("Lot size decrease", OBJ_LABEL, 0, 0, 0);
   ObjectSet("Lot size decrease", OBJPROP_XDISTANCE, 190);
   ObjectSet("Lot size decrease", OBJPROP_YDISTANCE, 103);
   ObjectSetText("Lot size decrease", "", 14, "Wingdings", Yellow);

   ObjectCreate("Help text", OBJ_LABEL, 0, 0, 0);
   ObjectSet("Help text", OBJPROP_XDISTANCE, 50);
   ObjectSet("Help text", OBJPROP_YDISTANCE, 140);
   ObjectSetText("Help text", "(Hold down Alt while double-clicking, or use keyboard shortcuts)", 8, "Arial", White);

   WindowRedraw();
}

// Delete the pseudo-button objects (on termination)
void RemoveObjects()
{
   ObjectDelete("Buy button");
   ObjectDelete("Sell button");
   ObjectDelete("Lot size");
   ObjectDelete("Lot size increase");
   ObjectDelete("Lot size decrease");
   ObjectDelete("Help text");
   WindowRedraw();
}

// Update the display of the lot size (in response to Ctrl+Shift+Plus etc)
void DisplayLotSize()
{
   ObjectSetText("Lot size", "Lots: " + DoubleToStr(glbLotSizeInUse, 2), 14, "Arial", Yellow);
   WindowRedraw();
}



// *********************************************************************
// Script loop - keeps running until you use right-click/Remove Script
// or until you press Ctrl+Shift+X
// *********************************************************************

int start()
{
   if (!IsDllsAllowed()) {
      MessageBox("You must turn on \'Allow DLL imports\' in order to use this script");
      return (0);
   }
   
   // Flag used to indicate whether Ctrl+Shift+X has been pressed   
   bool bContinue = true;
   
   // Keep looping forever until the script is removed from the 
   // chart, or Ctrl+Shift+X is pressed
   while (bContinue && !IsStopped()) {

      // Declare the variables which the DLL passes values back into.
      // Basically, if you pass an array of ints by reference in MQL,
      // the DLL sees an int*. Doesn't seem to be possible to pass
      // a single integer by reference - i.e. int & x, rather than int & x[]
      int EventType[1], Param1[1], Param2[1], Param3[1], EventLocalTime[1];
       
      // The DLL stores a queue of events for each window. Therefore, 
      // by the time this loop has done its sleep (below), there may then be
      // multiple events waiting to be processed. We can retrieve these
      // from the DLL simply by calling GetFromQueue() over and over again
      // until it fails, indicating that the queue is empty. If GetFromQueue()
      // returns true, then the description of the event is put into index 0 
      // of the array variables above.
      while (GetFromQueue(WindowHandle(Symbol(), 0), EventType, Param1, Param2, Param3, EventLocalTime)) {
         

         // Check what type of event we've got, as defined by EventType[0]. At the moment, the DLL
         // handles WM_KEYDOWN, WM_LBUTTONDBLCLK, and WM_RBUTTONDBLCLK. The parameters passed back
         // by the DLL are **NOT** the same as the actual window-message wParam and lParam parameters.
         // For WM_KEYDOWN, Param1[0] holds the virtual key code, Param2[0] is empty, and Param3[0]
         // contains flags indicating which control keys are being held down (see below).
         // For double-clicks, Param1[0] holds the x co-ordinate relative to the top of the window, 
         // Param2[0] holds the y co-ordinate, and Param3[0] again holds key flags.
         // The key flags in Param3[0] are a bit mask: bit zero is set if Shift is being held down,
         // bit 1 is set if Ctrl is being held down, and bit 2 is set if Alt is being held down.
         
         switch (EventType[0]) {

            // Handle key-down events in the chart window. (For reference: the DLL simply
            // doesn't seem to receive key presses which Metatrader handles itself, such
            // as Ctrl+G).
            case WM_KEYDOWN:
            
               // All our shortcut keys are Ctrl+Shift+something. Therefore, we require 
               // bits 0 and 1 to be set in the key-mask parameter (see above)
               if (Param3[0] == 3) {

                  // Handle the virtual key code which is being pressed
                  switch (Param1[0]) {
                     case 67:
                        // Ctrl+Shift+C                
                        CloseAllOrders();
                        break;
                     
                     case 66:
                        // Ctrl+Shift+B
                        DoBuy();
                        break;

                     case 83:
                        // Ctrl+Shift+S
                        DoSell();
                        break;

                     case 88:
                        // Ctrl+Shift+X
                        bContinue = false;
                        break;
                        
                     case 187:
                        // Ctrl+Shift+Plus
                        IncreaseLots();
                        break;

                     case 189:
                        // Ctrl+Shift+Minus
                        DecreaseLots();
                        break;
                  }               
               }
               break;


            // Handle double-clicks in the chart window
            case WM_LBUTTONDBLCLK:
            
               // As an accident-avoidance measure, we require the Alt key to be pressed while 
               // double-clicking. Therefore, we require bit 2 of the key-mask flag to be
               // set (see above).
               if (Param3[0] == 4) {

                  // Use the service function below to detect whether the x and y co-ordinates
                  // fall within any of the OBJ_LABEL objects we've created
                  if (IsMouseWithinObjectCoordinates(Param1[0], Param2[0], "Buy button")) {
                     DoBuy();
                  } else if (IsMouseWithinObjectCoordinates(Param1[0], Param2[0], "Sell button")) {
                     DoSell();
                  } else if (IsMouseWithinObjectCoordinates(Param1[0], Param2[0], "Lot size increase")) {
                     IncreaseLots();
                  } else if (IsMouseWithinObjectCoordinates(Param1[0], Param2[0], "Lot size decrease")) {
                     DecreaseLots();
                  }
               }               
               break;               
         }
      }

      // Sleep for a little bit - during which *multiple* mouse and keyboard events may happen,
      // and there may subsequently be multiple successful calls to GetFromQueue()
      Sleep(100);
   }

   // Either the script has been removed from the chart, or Ctrl+Shift+X has been pressed
   return(0);
}


// Service function for detecting whether X and Y co-ordinates (from a double-click)
// fall within an OBJ_LABEL which we've created on the chart. We can get Metatrader
// to tell us the top and left co-ordinates of the OBJ_LABEL (in case the user has 
// moved it from its position as set in CreateObjects), but we can't get the width
// and height. We have to cheat, and use hard-coded definitions based on the font size.
bool IsMouseWithinObjectCoordinates(int X, int Y, string ObjName)
{
   // Get the top left of the object 
   int ObjectX = ObjectGet(ObjName, OBJPROP_XDISTANCE);
   int ObjectY = ObjectGet(ObjName, OBJPROP_YDISTANCE);
   
   // Make sure that the co-ordinates of the double-click are beyond this
   if (X < ObjectX || Y < ObjectY) return (false);
   
   // Get the hard-coded definition of the object's width and height.
   // These hard-coded cheats require changing if you alter the font size
   // of the OBJ_LABEL objects 
   int width, height;
   if (ObjName == "Buy button") {
      width = 55;
      height = 30;
   } else if (ObjName == "Sell button") {
      width = 65;
      height = 30;
   } else if (ObjName == "Lot size increase") {
      width = 15;
      height = 20;
   } else if (ObjName == "Lot size decrease") {
      width = 15;
      height = 20;
   }

   // Is the location of the double-click outside the object?
   if (X > ObjectX + width) return (false);
   if (Y > ObjectY + height) return (false);

   // Yes! A double-click on the OBJ_LABEL
   return (true);
}



// *********************************************************************
// Functions which increase the number of lots used for buys and sells,
// either in response to Ctrl+Shift+plus/minus or in response to
// Alt+double-clicking on the up and down arrows
// *********************************************************************

void IncreaseLots()
{
   glbLotSizeInUse += LotSizeStep;
   DisplayLotSize();
}

void DecreaseLots()
{
   if (glbLotSizeInUse - LotSizeStep > 0) {
      glbLotSizeInUse -= LotSizeStep;
      DisplayLotSize();
   }
}



// *********************************************************************
// Trading functions. N.B. Very limited error handling. Not entirely
// suitable for real-life use. Could also modify these functions so that
// they use a specific magic number for trade placement, and so that
// CloseAllOrders() then only closes things placed by this code, rather 
// than all orders for the symbol
// *********************************************************************


// Close all open and pending orders for the chart's symbol
void CloseAllOrders()
{
   if (ConfirmActions) {
      if (MessageBox("Really close all open and pending orders for " + Symbol() + "?", "Close?", 36) == 7) return;
   }

   // Close all the orders. No retries if anything fails
   for (int i = OrdersTotal() - 1; i >=0 ; i--) {
      if (OrderSelect(i, SELECT_BY_POS)) {
         if (OrderSymbol() == Symbol()) {
            switch (OrderType()) {
               case OP_BUYLIMIT:
               case OP_BUYSTOP:
               case OP_SELLLIMIT:
               case OP_SELLSTOP:
                  OrderDelete(OrderTicket());
                  break;
            
               case OP_BUY:
                  RefreshRates();
                  OrderClose(OrderTicket(), OrderLots(), Bid, 999);
                  break;
               
               case OP_SELL:
                  RefreshRates();
                  OrderClose(OrderTicket(), OrderLots(), Ask, 999);
                  break;
            }
         }
      }
   }
}

// Place a buy order at market
void DoBuy()
{
   if (ConfirmActions) {
      if (MessageBox("Really buy " + DoubleToStr(glbLotSizeInUse, 2) + " lot(s) at market on " + Symbol() + "?", "Buy?", 36) == 7) return;
   }

   RefreshRates();
   if (OrderSend(Symbol(), OP_BUY, glbLotSizeInUse, Ask, 999, 0, 0) > 0) {
      // Okay
   } else {
      int err = GetLastError();
      MessageBox("Placement of buy order failed with error #" + err);
   }
}

// Place a sell order at market
void DoSell()
{
   if (ConfirmActions) {
      if (MessageBox("Really sell " + DoubleToStr(glbLotSizeInUse, 2) + " lot(s) at market on " + Symbol() + "?", "Sell?", 36) == 7) return;
   }
 
   RefreshRates();
   if (OrderSend(Symbol(), OP_SELL, glbLotSizeInUse, Bid, 999, 0, 0) > 0) {
      // Okay
   } else {
      int err = GetLastError();
      MessageBox("Placement of sell order failed with error #" + err);
   }
}




